% Function: This is the program that calculates hits using the alpha
% calculated based different consistency levels across multiple 
% experiments. The global consistent hits performs this for all comparisons
% as well as for subsets of the data

% Inputs:
% PValueCell - a cell array, where each entry is another cell array that 
%    contains the matrices with the significance values for a each 
%    comparison in a given experiment
% FCCell - a cell array, where each entry is another cell array that 
%    contains the matrices with the fold change values for a each 
%    comparison in a given experiment
% PCutoffCell - a cell array with the different alpha cutoff values for the
%    each possible consistency level
% PerAgreeCell - a cell that lists the number of experiments a IP/Probe
%    combination must be significant with the associated p-value cutoff in
%    order to satisfy the criteria to be a hit
% IPProbeList - a cell array that lists the name of each IP/Probe
%    combination
% CondNames - a cell array where each entry is the name of the condition or 
%    comparison that associates with the p-values and fold changes
% ExpName - a string name of the experiment to be used for saving files

% Outputs:
% HitsCells - a complicated cell array that contains information about the
%    identity, number, matrix index of the significant hits

function [HitCells] = GlobalConsistentHits(PValueCell,FCCell,...
    PCutoffCell,PerAgreeCell,IPProbeList,CondNames,ExpName)

CondNum = length(CondNames);
IPProbeNum = length(PValueCell{1}{1}(:));
[~,ExpNum] = size(PValueCell);
HitCells = cell(CondNum,1);

for alpha = 1:CondNum
    
    FCArray_a = zeros(IPProbeNum,ExpNum);
    PArray_a = zeros(IPProbeNum,ExpNum);
    
    for beta = 1:ExpNum
    
        FCArray_a(:,beta) = FCCell{beta}{alpha}(:);
        PArray_a(:,beta) = PValueCell{beta}{alpha}(:);
    end

    SigListCombCell = cell(length(PCutoffCell),1);
    SigNumCombCell = cell(length(PCutoffCell),1);
    SigIndCombCell = cell(length(PCutoffCell),1);
    SigIDCombCell = cell(length(PCutoffCell),1);
   

    for i = 1:ExpNum

        iCombinations = combnk(1:ExpNum,i);

        [nCombinations,~] = size(iCombinations);

        SigListCombCell_i = cell(nCombinations,1);
        SigNumCombCell_i = cell(nCombinations,1);
        SigIndCombCell_i = cell(nCombinations,1);
        SigIDCombCell_i = cell(nCombinations,1);

        for j = 1:nCombinations

            [SigIDCombCell_i{j},SigIndCombCell_i{j},...
                SigListCombCell_i{j},SigNumCombCell_i{j}] = ...
                SingleConsistentHits(FCArray_a(:,iCombinations(j,:)),...
                PArray_a(:,iCombinations(j,:)),PCutoffCell{i},...
                PerAgreeCell{i},IPProbeList); 
            %Function - Statistics/MetaConsistency

        end

        SigListCombCell{i} = SigListCombCell_i;
        SigNumCombCell{i} =  SigNumCombCell_i;
        SigIndCombCell{i} = SigIndCombCell_i;
        SigIDCombCell{i} = SigIDCombCell_i;

    end

%     Commented out code. This code was used to monitor how the number of
%     statistically significant hits changed with increasing experimental
%     N.
%     figure(01)
%     hold on
%     for gamma = 1:ExpNum
% 
%        L_comb = length(SigNumCombCell{gamma});
% 
%        for delta = 1:L_comb
% 
%            plot(gamma,SigNumCombCell{gamma}{delta}(1),'o',...
%                'linewidth',3,'markersize',15)
%            
%        end
%     end
%     title(strcat('Graph of consistent hits for',{' '},ExpName,...
%    'between',{' '},CondNames{alpha},{' '},...
%    'over all experiment combinations'))
%     saveas(01,strcat(ExpName,CondNames{alpha},'ConsistentHits','.fig'),...
%         'fig');
%     close(gcf)
%     
%     color = hot(ExpNum);
%     figure(02)
%     hold on
%     for epsilon = 1:ExpNum
% 
%        L_comb = length(SigNumCombCell{epsilon});
% 
%        for kappa = 1:L_comb
% 
%            for lambda = 1:length(SigNumCombCell{epsilon}{kappa})
% 
%                per_k = (length(SigNumCombCell{epsilon}{kappa})-lambda+1)/...
%                    length(SigNumCombCell{epsilon}{kappa});
% 
%                plot(per_k,SigNumCombCell{epsilon}{kappa}(lambda),'o',...
%                    'color',color(epsilon,:),'linewidth',3,...
%                    'markersize',15)
%            end
%        end
%     end
%    title(strcat('Graph of consistent hits with unions for',...
%    {' '},ExpName,'between',{' '},CondNames{alpha},{' '},...
%    'over all experiment combinations'))
%     saveas(02,strcat(ExpName,CondNames{alpha},'UnionConsistentHits',...
%         '.fig'),'fig');
%     close(gcf)
    
    HitCells{alpha} = {SigIDCombCell;SigListCombCell;SigNumCombCell;...
        SigIndCombCell};
    
end